#include "SSH.h"

#ifdef PLATFORM_WIN32
#pragma comment(lib, "ws2_32.lib")
#endif

namespace Upp {

#define LLOG(x)	 RLOG(x)

// U++ style memory (de/re)allocators.
extern void* SshAlloc(size_t size, void **abstract);
extern void* SshRealloc(void *ptr, size_t size, void **abstract);
extern void  SshFree(void *ptr, void **abstract);

Ssh& Ssh::StartConnect(const String& host, int port)
{
	// Init
	AddJob() << [=] {
		if(host.IsEmpty())
			Error(-1, t_("Host is not specified"));
		else
		if(username.IsEmpty() || password.IsEmpty())
			Error(-1, t_("Username or password is not specified."));
		socket.Clear();
		socket.WhenWait = Proxy(WhenDo);
		session = NULL;
		ip_addrinfo.Start(host, port);
		LLOG(Format("** SSH2: Starting DNS sequence locally for %s:%d", host, port));
		return false;
	};

	// DNS lookup & socket creation.
	AddJob() << [=] {
		if(ip_addrinfo.InProgress())
			return true;
		if(!ip_addrinfo.GetResult())
			Error(-1, Format(t_("DNS query for '%s' failed."), host));
		return false;
	};

	// Connect to SSH server and init libssh2 session
	AddJob() << [=] {
		if(socket.Connect(ip_addrinfo) && socket.WaitConnect()) {
			LLOG(Format("++ SSH2: Successfully connected to %s:%d", host, port));
			ip_addrinfo.Clear();
			// Maybe we should make the memory managers (system/upp) switchable on runtime?
#ifdef flagUSEMALLOC
			LLOG("** SSH2: Using system memory manager (malloc).");
			session = libssh2_session_init();
#else
			LLOG("** SSH2: Using U++ style memory managers.");
			session = libssh2_session_init_ex((*SshAlloc), (*SshFree), (*SshRealloc), NULL);
#endif
			if(!session)
				Error(-1, t_("Unable to initalize libssh2 session."));
			libssh2_session_set_blocking(session, 0L);
			return false;
		}
		if(!socket.IsError())
			Error(-1, t_("Couldn't connect to ") << host);
		return true;
	};

	// Start handshake
	AddJob() << [=] {
		auto rc = libssh2_session_handshake(session, socket.GetSOCKET());
		if(rc == 0) {
			LLOG("++ SSH2: Handshake succesfull.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};

	// Get host key hash (fingerprint)
	AddJob() << [=] {
		fingerprint = libssh2_hostkey_hash(session, LIBSSH2_HOSTKEY_HASH_SHA1);
		if(fingerprint.IsEmpty())
			LLOG("!! SSH2: Fingerprint is not available.");
		WhenHash();
		DUMPHEX(fingerprint);
		return false;
	};

	// Retrieve authorization methods
	AddJob() << [=] {
		auth_methods = libssh2_userauth_list(session, username, username.GetLength());
		if(!auth_methods.IsEmpty()) {
			LLOG("++ SSH2: Authentication methods successfully retrieved.");
			return false;
		}
		if(!WouldBlock()) Error();
		return true;
	};

	// Authenticate user
	AddJob() << [=] {
		auto rc = libssh2_userauth_password(session, username, password);
		if(rc == 0) {
			LLOG("++ SSH2: Client succesfully authenticated and connected.");
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

Ssh& Ssh::StartDisconnect()
{
	// Stop SSH session
	AddJob() << [=] {
		if(session) {
			int rc = libssh2_session_disconnect(session, "Disconnecting...");
			if(rc == 0) return false;
			return rc == LIBSSH2_ERROR_EAGAIN;
		}
		return false;
	};
	// Free libssh2 session handle
	AddJob() << [=] {
		if(!session) {
			LLOG("!! SSH2: No session handle found. Couldn't release resources.");
			return false;
		}
		auto rc = libssh2_session_free(session);
		if(rc == 0) {
			LLOG("++ SSH2: Client disconnected, and resources released.");
			session = NULL;
			return false;
		}
		if(rc != LIBSSH2_ERROR_EAGAIN) Error();
		return true;
	};
	return *this;
}

void Ssh::CleanUp()
{
	if(session) StartDisconnect();
	if(!IsCleanup()) Execute();
	else LLOG("** SSH: Performing clean up...");
}

void Ssh::Error(int code, const char* msg)
{
	Tuple<int, String> t = Ssh2LibError(session, code, msg);
	Halt(t.Get<int>(), t.Get<String>());
}

Ssh::Ssh()
{
	session = NULL;
	GlobalTimeout(60000);
	WhenHalt = [=] { CleanUp(); };
}

SshSubsystem::SshSubsystem()
{
	type = Type::UNDEFINED;
	ssh = NULL;
	chunk_size = int(65536);
	packet_length = int64(0);
	WhenHalt = [=] { CleanUp(); };
}

void SshSubsystem::Session(Ssh& session)
{
	ssh = &session;
	// Redirect all subsystems' WhenDo calls to the session's defined method.
	// This is default. But defining WhenDo method per subsysetem instance is also possible.
	WhenDo = [=] { ssh->WhenDo(); };
	// Schedule init for the given subsystem.
	StartInit();
}

void SshSubsystem::Error(int code, const char* msg)
{
	ASSERT(ssh);
	auto t = Ssh2LibError(ssh->GetSession(), code, msg);
	Halt(t.Get<int>(), t.Get<String>());
}

Tuple<int, String> Ssh2LibError(SshSession* session, int code, const char* msg)
{
	Tuple<int, String> t;
	if(session && code == 0) {
		Buffer<char*> libmsg(512);
		int rc = libssh2_session_last_error(session, libmsg, NULL, 0);
		t = MakeTuple<int, String>(rc, *libmsg);
	}
	else
		t = MakeTuple<int, String>(code, msg);
	LLOG(Format("-- SSH2: Error (code: %d): %s", t.Get<int>(), t.Get<String>()));
	return pick(t);
}

INITIALIZER(SSH) {
	LLOG("Initializing libssh2...");
	libssh2_init(0);
}
EXITBLOCK {
	LLOG("Deinitializing libssh2...");
	libssh2_exit();
}
}